const API_BASE = 'https://123share.cn'; // 替换你的域名

// ==========================================
// 1. 初始化菜单
// ==========================================
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: "share-selection",
    title: "发送选中文字 到 极简快传",
    contexts: ["selection"]
  });
  chrome.contextMenus.create({
    id: "share-image",
    title: "发送此图片 到 极简快传",
    contexts: ["image"]
  });
});

// ==========================================
// 2. 菜单点击监听 (核心入口)
// ==========================================
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  // 提示：我们移除了原先的 "Processing" 系统通知，以免在 macOS 上造成阻塞或困惑
  // 直接开始执行逻辑

  // 预先确保 Offscreen 文档已创建
  try {
    await setupOffscreenDocument('offscreen.html');
  } catch (err) {
    console.warn('Offscreen document setup warning:', err);
  }

  if (info.menuItemId === "share-selection" && info.selectionText) {
    handleSendText(info.selectionText);
  } 
  else if (info.menuItemId === "share-image" && info.srcUrl) {
    handleSendImage(info.srcUrl);
  }
});

// ==========================================
// 3. 核心逻辑：发送文本
// ==========================================
async function handleSendText(text) {
  try {
    const res = await fetch(`${API_BASE}/api/share/text`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ text: text, burn: false })
    });
    const data = await res.json();

    if (data.code) {
      await handleSuccess(data.code, '文本');
    } else {
      throw new Error(data.error || '未知错误');
    }
  } catch (err) {
    handleError('文本发送失败', '网络连接超时或服务器繁忙');
  }
}

// ==========================================
// 4. 核心逻辑：发送图片
// ==========================================
async function handleSendImage(imageUrl) {
  try {
    // Fetch 图片 (增加 15秒 超时控制)
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 15000); // 15秒超时

    const imgRes = await fetch(imageUrl, { signal: controller.signal });
    clearTimeout(timeoutId);

    if (!imgRes.ok) throw new Error(`无法下载图片 (${imgRes.status})`);
    
    const imgBlob = await imgRes.blob();

    // 智能推断文件名
    let filename = 'image.jpg';
    try {
        const urlPath = new URL(imageUrl).pathname;
        const extractedName = urlPath.split('/').pop();
        if (extractedName && extractedName.includes('.')) {
            filename = extractedName.split(/[#?]/)[0];
        } else {
            const mime = imgBlob.type;
            if (mime === 'image/png') filename = `img_${Date.now()}.png`;
            else if (mime === 'image/gif') filename = `img_${Date.now()}.gif`;
            else if (mime === 'image/webp') filename = `img_${Date.now()}.webp`;
        }
    } catch(e) {}

    const formData = new FormData();
    formData.append('file', imgBlob, filename);
    formData.append('burn', 'false');

    const uploadRes = await fetch(`${API_BASE}/api/share/file`, {
      method: 'POST',
      body: formData
    });
    const data = await uploadRes.json();

    if (data.code) {
      await handleSuccess(data.code, '图片');
    } else {
      throw new Error(data.error || '上传失败');
    }

  } catch (err) {
    console.error(err);
    // 区分超时错误
    const msg = err.name === 'AbortError' ? '图片下载超时，请检查网络' : (err.message || '网络错误');
    handleError('图片发送失败', msg);
  }
}

// ==========================================
// 5. 辅助函数 (成功与错误处理 - 改用注入式弹窗)
// ==========================================

async function handleSuccess(code, type) {
  const copyContent = `【极简快传】您收到一个${type}\n取件码：${code}\n请在小程序或网页(${API_BASE})提取`;

  // 1. 写剪贴板 (Fire-and-forget，不阻塞后续弹窗)
  addToClipboard(copyContent).catch(err => console.log('剪贴板写入异常:', err));

  // 2. 在当前网页弹出成功提示
  showPageToast('success', `${type}已发送，`, `取件码 ${code} 已复制到剪贴板！`);
}

function handleError(title, message) {
  console.error(title, message);
  // 在当前网页弹出错误提示
  showPageToast('error', title, message);
}

// ==========================================
// 6. 核心功能：网页内弹窗注入 (Content Script Injection)
// ==========================================

async function showPageToast(type, title, msg) {
  try {
    // 获取当前活动标签页
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (tab && tab.id) {
      // 检查 URL 是否允许注入 (避免在 chrome:// 页面报错)
      if (tab.url.startsWith('chrome://') || tab.url.startsWith('edge://')) {
          // 如果是无法注入的页面，回退到系统通知
          fallbackSystemNotification(type, title, msg);
          return;
      }

      // 向当前网页注入 JS 代码
      await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: injectToastDom, // 执行下方的 DOM 操作函数
        args: [type, title, msg] // 传递参数
      });
    }
  } catch (err) {
    console.error('弹窗注入失败:', err);
    fallbackSystemNotification(type, title, msg);
  }
}

// 兜底方案：如果注入失败，还是尝试弹一下系统通知
function fallbackSystemNotification(type, title, msg) {
    chrome.notifications.create('fallback-' + Date.now(), {
      type: 'basic',
      iconUrl: 'icons/icon48.png',
      title: title,
      message: msg,
      priority: 2
    });
}

// 这个函数会被序列化并在目标网页的上下文中执行 (注意：这里不能引用外部变量)
function injectToastDom(type, title, msg) {
  // 1. 创建容器
  const div = document.createElement('div');
  div.id = 'fastshare-toast-' + Date.now();
  div.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 2147483647; /* 保证最顶层 */
    background: ${type === 'success' ? '#f0fdf4' : '#fef2f2'};
    border: 1px solid ${type === 'success' ? '#bbf7d0' : '#fecaca'};
    border-left: 5px solid ${type === 'success' ? '#22c55e' : '#ef4444'};
    border-radius: 8px;
    padding: 16px 20px;
    box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
    min-width: 300px;
    opacity: 0;
    transform: translateY(-20px);
    transition: all 0.3s ease-out;
    display: flex;
    flex-direction: column;
    pointer-events: none; /* 防止遮挡点击 */
  `;

  // 2. 填充内容
  const titleEl = document.createElement('div');
  titleEl.innerText = title;
  titleEl.style.cssText = `
    color: ${type === 'success' ? '#166534' : '#991b1b'};
    font-weight: 600;
    font-size: 16px;
    margin-bottom: 4px;
    text-align: left;
  `;

  const msgEl = document.createElement('div');
  msgEl.innerText = msg;
  msgEl.style.cssText = `
    color: ${type === 'success' ? '#15803d' : '#b91c1c'};
    font-size: 14px;
    text-align: left;
  `;

  div.appendChild(titleEl);
  div.appendChild(msgEl);

  // 3. 插入页面
  document.body.appendChild(div);

  // 4. 动画显示
  requestAnimationFrame(() => {
    div.style.opacity = '1';
    div.style.transform = 'translateY(0)';
  });

  // 5. 定时销毁 (3.5秒后)
  setTimeout(() => {
    div.style.opacity = '0';
    div.style.transform = 'translateY(-20px)';
    setTimeout(() => {
      if (div.parentNode) div.parentNode.removeChild(div);
    }, 300);
  }, 4000);
}

// ==========================================
// 7. 核心修复：更健壮的 Offscreen 管理
// ==========================================

let creating; // 全局单例锁

// 确保 Offscreen 文档存在的函数 (单例模式，防止重复创建)
async function setupOffscreenDocument(path) {
  // 1. 检查是否存在
  const existingContexts = await chrome.runtime.getContexts({
    contextTypes: ['OFFSCREEN_DOCUMENT']
  });
  if (existingContexts.length > 0) return;

  // 2. 如果正在创建，等待
  if (creating) {
    await creating;
  } else {
    // 3. 创建 (注意 reasons 必须包含 CLIPBOARD)
    creating = chrome.offscreen.createDocument({
      url: path,
      reasons: ['CLIPBOARD'], // 必须大写
      justification: 'Write text to clipboard'
    });
    await creating;
    creating = null;
  }
}

// 写入剪贴板函数
async function addToClipboard(text) {
    try {
        // 1. 确保文档存在
        await setupOffscreenDocument('offscreen.html');
        
        // 2. 发送消息
        await chrome.runtime.sendMessage({
            type: 'copy-data-to-clipboard',
            target: 'offscreen-doc',
            data: text
        });
    } catch (e) {
        console.error('Clipboard error:', e);
    }
}